<?php
require_once __DIR__ . '/config.php';
require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/db.php';
header('Content-Type: application/json; charset=utf-8');

// Set execution time limit
set_time_limit(30);

// Check if user is logged in
if(!isset($_SESSION['user'])){
    http_response_code(401);
    echo json_encode(['ok'=>false, 'error'=>'not authorized']); exit;
}

// Check if user is admin
$admins = ADMIN_LOGINS;
if(!in_array($_SESSION['user']['login'] ?? '', $admins, true)){
    http_response_code(403);
    echo json_encode(['ok'=>false, 'error'=>'forbidden']); exit;
}

// Check if attachment ID is provided
if(!isset($_POST['attachment_id']) || !is_numeric($_POST['attachment_id'])){
    http_response_code(400);
    echo json_encode(['ok'=>false, 'error'=>'invalid attachment id']); exit;
}

$attachment_id = intval($_POST['attachment_id']);

// Get attachment details from database with proper locking to prevent race conditions
try {
    $db->beginTransaction();
    
    // Lock the row for update
    $stmt = $db->prepare("SELECT * FROM attachments WHERE id = ? FOR UPDATE");
    $stmt->execute([$attachment_id]);
    $attachment = $stmt->fetch();
    
    if(!$attachment){
        $db->rollBack();
        http_response_code(404);
        echo json_encode(['ok'=>false, 'error'=>'attachment not found']); exit;
    }
    
    // Delete record from database first
    $stmt = $db->prepare("DELETE FROM attachments WHERE id = ?");
    $stmt->execute([$attachment_id]);
    
    // Commit the transaction
    $db->commit();
    
    // Delete file from filesystem after database deletion
    $filepath = UPLOAD_DIR . '/' . $attachment['filename'];
    if(file_exists($filepath)){
        // Use unlink with error suppression and verification
        $deleted = @unlink($filepath);
        if (!$deleted && file_exists($filepath)) {
            // Log the error but don't fail the operation
            error_log("Failed to delete file: " . $filepath);
        }
    }
    
    echo json_encode(['ok'=>true, 'message'=>'attachment deleted successfully']);
} catch (Exception $e) {
    $db->rollBack();
    http_response_code(500);
    echo json_encode(['ok'=>false, 'error'=>'database error: ' . $e->getMessage()]); exit;
}